//+------------------------------------------------------------------+
//|                                        AutoEncoder Indicator.mq5 |
//|                                     Copyright 2023, Omega Joctan |
//|                        https://www.mql5.com/en/users/omegajoctan |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Omega Joctan"
#property link      "https://www.mql5.com/en/users/omegajoctan"
#property version   "1.00"

//Load both the encoder_model and the decoder_model
#resource "\\Files\\encoder.eurusd.h1.onnx" as uchar encoder_onnx[];
#resource "\\Files\\decoder.eurusd.h1.onnx" as uchar decoder_onnx[];

// Load the MinMax scaler also
#resource "\\Files\\minmax_min.bin" as double min_values[];
#resource "\\Files\\minmax_max.bin" as double max_values[];


#include <Autoencoder-onnx.mqh>
#include <preprocessing.mqh>

CAutoEncoderONNX encoder_model; //for the encoder model
CAutoEncoderONNX decoder_model; //for the decoder model
MinMaxScaler *scaler; //Python-like MinMax scaler

#property indicator_chart_window
#property indicator_plots 1
#property indicator_buffers 5

input bool show_bars = true;
input bool show_bullish_bearish = true;

//--- plot Candle
#property indicator_label1  "autoencoded open; high; low; close"
#property indicator_type1   DRAW_COLOR_CANDLES
#property indicator_color1  clrRed, clrGray
#property indicator_style1  STYLE_SOLID
#property indicator_width1  1

double open_candle[];
double high_candle[];
double low_candle[];
double close_candle[];
double color_buffer[];
//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
  {
  
  long bg = clrBlack;
  ChartSetInteger(0, CHART_COLOR_BACKGROUND, bg);
  ChartSetInteger(0, CHART_COLOR_FOREGROUND, clrWhite);
  
  if (!show_bars)
    {
      ChartSetInteger(0, CHART_COLOR_CANDLE_BEAR, bg);
      ChartSetInteger(0, CHART_COLOR_CANDLE_BULL, bg);      
      ChartSetInteger(0, CHART_COLOR_CHART_UP, bg);
      ChartSetInteger(0, CHART_COLOR_CHART_DOWN, bg);
      ChartSetInteger(0, CHART_SHOW_GRID, false);
      ChartSetInteger(0, CHART_SHOW_PERIOD_SEP, true);
      ChartSetInteger(0, CHART_SHOW_ASK_LINE, true);
      ChartSetInteger(0, CHART_SHOW_BID_LINE, true);
      ChartSetInteger(0, CHART_COLOR_ASK, clrRed);
      ChartSetInteger(0, CHART_COLOR_BID, clrRed);
      ChartSetInteger(0, CHART_SHOW_VOLUMES, false);
      ChartSetInteger(0, CHART_COLOR_CHART_LINE, bg);
    }
   else
     {
      ChartSetInteger(0, CHART_COLOR_CANDLE_BEAR, clrBlack);
      ChartSetInteger(0, CHART_COLOR_CANDLE_BULL, clrPink);      
      ChartSetInteger(0, CHART_COLOR_CHART_UP, clrPink);
      ChartSetInteger(0, CHART_COLOR_CHART_DOWN, clrPink);
      ChartSetInteger(0, CHART_SHOW_GRID, false);
      ChartSetInteger(0, CHART_SHOW_PERIOD_SEP, true);
      ChartSetInteger(0, CHART_SHOW_ASK_LINE, true);
      ChartSetInteger(0, CHART_SHOW_BID_LINE, true);
      ChartSetInteger(0, CHART_COLOR_ASK, clrPink);
      ChartSetInteger(0, CHART_COLOR_BID, clrPink);
      ChartSetInteger(0, CHART_SHOW_VOLUMES, false);
      ChartSetInteger(0, CHART_COLOR_CHART_LINE, clrPink);
     }
  
//--- indicator buffers mapping
   
   ChartSetSymbolPeriod(0, Symbol(), PERIOD_H1);
   
   SetIndexBuffer(0,open_candle,INDICATOR_DATA);
   SetIndexBuffer(1,high_candle,INDICATOR_DATA);
   SetIndexBuffer(2,low_candle,INDICATOR_DATA);
   SetIndexBuffer(3,close_candle,INDICATOR_DATA);
   
   SetIndexBuffer(4,color_buffer, INDICATOR_COLOR_INDEX);

//---
   
   PlotIndexSetDouble(0, PLOT_EMPTY_VALUE, 0.0);
   PlotIndexSetDouble(1, PLOT_EMPTY_VALUE, 0.0);
   PlotIndexSetDouble(2, PLOT_EMPTY_VALUE, 0.0);
   PlotIndexSetDouble(3, PLOT_EMPTY_VALUE, 0.0);
   
   IndicatorSetInteger(INDICATOR_DIGITS, _Digits);
//---
   IndicatorSetString(INDICATOR_SHORTNAME, "Autoencoder "+EnumToString(PERIOD_CURRENT));   
   ChartRedraw();   
//---

   if (!encoder_model.Init(encoder_onnx)) //initializing the encoder
     return INIT_FAILED;
   
   if (!decoder_model.Init(decoder_onnx)) //initializing the decoder
     return INIT_FAILED;
   
   scaler = new MinMaxScaler(min_values, max_values); //Load the Minmax scaler saved in python
     
//---
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime& time[],
                const double& open[],
                const double& high[],
                const double& low[],
                const double& close[],
                const long& tick_volume[],
                const long& volume[],
                const int& spread[])
  {
//---
   
   int start = prev_calculated;
   if(start>=rates_total)
      start = rates_total-1;
   
   
   vector encoded_data = {}, decoded_data = {};
   for(int i = start; i<rates_total; i++)
     {
        vector x_inputs = {open[i], high[i], low[i], close[i]};
        
        x_inputs = scaler.transform(x_inputs); //Normalize the input data, important!
        encoded_data = encoder_model.predict(x_inputs); //encode the data
        decoded_data = decoder_model.predict(encoded_data); //decode the data
        
        decoded_data = scaler.inverse_transform(decoded_data); //return data to its original state  
          
        open_candle[i]= decoded_data[0];
        high_candle[i]= decoded_data[1];
        low_candle[i]=  decoded_data[2];
        close_candle[i]=decoded_data[3];
        
        // Set upper and lower body colors based on the gradient
        
        if (close_candle[i]>open_candle[i])
         {
           color_buffer[i] = 1.0; //Draw gray for bullish candle
           close_candle[i] = high_candle[i];
           open_candle[i] = low_candle[i];
         }
        else
         {
           color_buffer[i] = 0.0; //draw red when there was a bearish candle
          
           close_candle[i] = low_candle[i];
           open_candle[i] = high_candle[i];
         }
         
        //--- Show bearish and bullish candles based on actual open and close prices
         
         if (show_bullish_bearish)
          {
           if (close[i]>open[i])
            color_buffer[i] = 1.0;
           else
             color_buffer[i] = 0.0;
          }
                         
        if (MQLInfoInteger(MQL_DEBUG))
         Comment(StringFormat("plotting [%d/%d] OPEN[%.5f] HIGH[%.5f] LOW[%.5f] CLOSE[%.5f]",i,rates_total,open_candle[i],high_candle[i],low_candle[i],close_candle[i]));
     }
     
//--- return value of prev_calculated for next call
   return(rates_total);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
     if (CheckPointer(scaler)!=POINTER_INVALID)
       delete scaler;
       
       
      ChartSetInteger(0, CHART_COLOR_CANDLE_BEAR, clrBlack);
      ChartSetInteger(0, CHART_COLOR_CANDLE_BULL, clrPink);      
      ChartSetInteger(0, CHART_COLOR_CHART_UP, clrPink);
      ChartSetInteger(0, CHART_COLOR_CHART_DOWN, clrPink);
      ChartSetInteger(0, CHART_SHOW_GRID, false);
      ChartSetInteger(0, CHART_SHOW_PERIOD_SEP, true);
      ChartSetInteger(0, CHART_SHOW_ASK_LINE, true);
      ChartSetInteger(0, CHART_SHOW_BID_LINE, true);
      ChartSetInteger(0, CHART_COLOR_ASK, clrPink);
      ChartSetInteger(0, CHART_COLOR_BID, clrPink);
      ChartSetInteger(0, CHART_SHOW_VOLUMES, false);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+

